import React, { useState, useEffect, useCallback, useRef } from 'react';
import axios from 'axios';

const SearchPage = ({
  rowsPerPage = 10,
  fetchDataUrl,
  itemKeyMapping,
  ListComponent,
  actionFormats = [],
  additionalParams = {},
  identifierKey,
  onActionClick, // for popup actions
  dataKey = 'items',
  refreshKey,
  // Array of search field definitions.
  // Each field can be defined as:
  // { name, type, placeholder, defaultValue, (for dropdowns) options: [{label, value}, ...] }
  searchFields = [],
}) => {
  // If no searchFields are passed, default to a text search field "q"
  const effectiveSearchFields = (searchFields && searchFields.length > 0)
    ? searchFields
    : [{
        name: 'q',
        type: 'text',
        placeholder: 'Search...',
        defaultValue: '',
      }];

  // Initialize state for search filters using effectiveSearchFields
  const [searchFilters, setSearchFilters] = useState(() => {
    const initial = {};
    effectiveSearchFields.forEach(field => {
      initial[field.name] = field.defaultValue || '';
    });
    return initial;
  });

  const [items, setItems] = useState([]);
  const [isLoading, setIsLoading] = useState(false);
  const [page, setPage] = useState(0);
  const [totalCount, setCount] = useState(0);
  const [shouldFetchMore, setShouldFetchMore] = useState(true);
  const memoizedActionClick = useCallback(onActionClick, []);
  const initialRender = useRef(true);

  // Update a given filter's value.
  const handleFilterChange = (fieldName, value) => {
    setSearchFilters(prev => ({
      ...prev,
      [fieldName]: value,
    }));
  };

  // Merge additionalParams, pagination info, and search filters into one query object.
  const buildQueryParams = () => ({
    ...additionalParams,
    skip: page * rowsPerPage,
    limit: rowsPerPage,
    ...searchFilters,
  });

  const fetchItems = async () => {
    if (isLoading || (!shouldFetchMore && page > 0)) return;
    setIsLoading(true);
    try {
      const response = await axios.get(fetchDataUrl, { params: buildQueryParams() });
      // Assume that the response contains an array of items under dataKey.
      console.log(response.data);
      const fetchedItems = Array.isArray(response.data[dataKey])
        ? response.data[dataKey]
        : [];
      
      // If it's the first page, replace existing items; otherwise, append.
      if (page === 0) {
        setItems(fetchedItems);
      } else {
        setItems(prev => [...prev, ...fetchedItems]);
      }

      if ('total_count' in response.data) {
        setCount(response.data.total_count);
      }
      // If we received a full page, allow fetching more.
      setShouldFetchMore(fetchedItems.length === rowsPerPage);
    } catch (error) {
      console.error("Error fetching items:", error);
    } finally {
      setIsLoading(false);
    }
  };

  const handleLoadMore = () => {
    if (!isLoading && shouldFetchMore) {
      setPage(prev => prev + 1);
    }
  };

  // When filters change, reset pagination and fetch new results.
  const resetAndFetch = () => {
    setPage(0);
    setCount(0);
    setShouldFetchMore(true);
    fetchItems();
  };

  // Trigger fetch on page changes.
  useEffect(() => {
    if (initialRender.current) {
      initialRender.current = false;
    } else {
      fetchItems();
    }
  }, [page]);

  // Optionally refetch when refreshKey changes.
  useEffect(() => {
    if (!initialRender.current) {
      resetAndFetch();
    }
  }, [refreshKey]);

  // Listen for Enter key to trigger search.
  const handleKeyDown = (e) => {
    if (e.key === "Enter") {
      resetAndFetch();
    }
  };

  const handleScrollToTop = () => {
    document.documentElement.scrollTop = 0;
    document.body.scrollTop = 0;
  };

  return (
    <div className='projects-list-container'>
      <div className="search-and-total" id={identifierKey}>
        <div className="search-fields">
          {effectiveSearchFields.map(field => {
            switch (field.type) {
              case "text":
                return (
                  <input
                    key={field.name}
                    type="text"
                    placeholder={field.placeholder || field.name}
                    value={searchFilters[field.name]}
                    onChange={(e) => handleFilterChange(field.name, e.target.value)}
                    onKeyDown={handleKeyDown}
                    className="search-box"
                  />
                );
              case "date":
                return (
                  <input
                    key={field.name}
                    type="date"
                    placeholder={field.placeholder || field.name}
                    value={searchFilters[field.name]}
                    onChange={(e) => handleFilterChange(field.name, e.target.value)}
                    onKeyDown={handleKeyDown}
                    className="search-box"
                  />
                );
              case "time":
                return (
                  <input
                    key={field.name}
                    type="time"
                    placeholder={field.placeholder || field.name}
                    value={searchFilters[field.name]}
                    onChange={(e) => handleFilterChange(field.name, e.target.value)}
                    onKeyDown={handleKeyDown}
                    className="search-box"
                  />
                );
              case "dropdown":
                return (
                  <select
                    key={field.name}
                    value={searchFilters[field.name]}
                    onChange={(e) => handleFilterChange(field.name, e.target.value)}
                    onKeyDown={handleKeyDown}
                    className="search-dropdown"
                  >
                    {field.options.map(option => (
                      <option key={option.value} value={option.value}>
                        {option.label}
                      </option>
                    ))}
                  </select>
                );
              default:
                return (
                  <input
                    key={field.name}
                    type="text"
                    placeholder={field.placeholder || field.name}
                    value={searchFilters[field.name]}
                    onChange={(e) => handleFilterChange(field.name, e.target.value)}
                    onKeyDown={handleKeyDown}
                    className="search-box"
                  />
                );
            }
          })}
        </div>
        <span className="total-projects">{totalCount} items in total</span>
        <button onClick={resetAndFetch}>Search</button>
      </div>
      <div className="list-content">
        <ListComponent
          items={items}
          mapping={itemKeyMapping}
          onAddClick={() => {}}
          headers={Object.keys(itemKeyMapping)}
          actionFormats={actionFormats}
          identifierKey={identifierKey}
          onActionClick={memoizedActionClick}
          dataKey={dataKey}
        />
        {shouldFetchMore && (
          <div className='load-more-container'>
            <a onClick={handleLoadMore} disabled={isLoading} className='load-more'>
              Load More
              <span className="material-symbols-outlined">navigate_next</span>
            </a>
            <a href={`#${identifierKey}`} className="back-to-top">
              ↑ Top
            </a>
          </div>
        )}
      </div>
      {items.length === 0 && !isLoading && <p>No items found.</p>}
    </div>
  );
};

export default SearchPage;
